function [] = aratome2freeframe(varargin)
%This function converts aratome stitched data back into the freeframe
%directory structure it originally came from. It is expected that the
%aratome structure is \root\session\sitiched\...files
%Synatax:   [] = aratome2freeframe('overlap','10','date','1976-11-01');
%Input:     'ardir' = the directory of stitiched data.
%           'ordir' = the original directory structure
%Output:    no output at this moment

[ardir,ordir] = parse(varargin);

%where are the files?
if isempty(ardir)
    ardir = uigetdir2('','Aratome Stitiched Image Directory');    %get the directory
end
if isempty(ordir)
    ordir = uigetdir2('','Original Freeframe directory structure'); 
end
%get the root directory because we are going to save to it
sepidx = strfind(ardir,filesep);
root_dir = ardir(1:sepidx(end)-1);  %no filesep in the root dir end
dir_name = ardir(sepidx(end)+1:end);  %grab the name of the current directory name
%make the converted save file folder
mkdir(root_dir,'AratomeStitched');
sav_dir = [root_dir,filesep,'AratomeStitched'];
mkdir(sav_dir,dir_name);
sav_dir = [sav_dir,filesep,dir_name];

%now lets get to the files
%original directory structure
orsessions = dir(ordir);
dir_idx = [orsessions.isdir];   %grab all of the isdir numbers
dir_idx(1:2) = 0;               %the first two directories are not what we want
session_names = {orsessions.name};   %grab the all of the names in the root
session_names = session_names(dir_idx);    %the directory names in root
%aratome directory structure
arsessions = dir(ardir);
dir_idx = [arsessions.isdir];   %grab all of the isdir numbers
dir_idx(1:2) = 0;               %the first two directories are not what we want
arsession_names = {arsessions.name};   %grab the all of the names in the root
arsession_names = arsession_names(dir_idx);    %the directory names in root

%prepare for resort files and save out
for i = 1:size(session_names,2)  %these are the sessions
    %now grab the images in aratome stitched directory
    curr_ardir = [ardir,filesep,arsession_names{i},filesep,'stitched'];     %current session step one down to stitched
    dir_info = dir(curr_ardir); %grab the current dir
    curr_names = {dir_info.name};   %grab the all of the names
    idx = [dir_info.isdir];   %grab all of the isdir numbers
    curr_files = curr_names(~idx);    %the files
    
    %now grab the original directory structure directory
    curr_ordir = [ordir,filesep,session_names{i}];     %current session step one down to stitched
    dir_info = dir(curr_ordir); %grab the current dir
    curr_names = {dir_info.name};   %grab the all of the names
    idx = [dir_info.isdir];   %grab all of the isdir numbers
    idx(1:2) = 0;   %toss the first two
    chan_dir = curr_names(idx);    %the channel directories

    %we now need to match channel to directory name
    [sortedfiles,sorteddir,ch_num,slice_num]= ch2dir(chan_dir,curr_files);
    
    %now sort the files and save out
    mkdir(sav_dir,session_names{i})  %make the current save out folder
    cur_sav_dir = [sav_dir,filesep,session_names{i}];
    
    %now grab the files
    for j = 1:slice_num
        for k = 1:ch_num  %cycle through files
            if j==1  %if on the first slice
                mkdir(cur_sav_dir,sorteddir{k})  %make the current save out folder
            end
            %copy over the file
            copyfile([curr_ardir,filesep,sortedfiles{j,k}],[cur_sav_dir,filesep,sorteddir{k},filesep,sortedfiles{j,k},'.tif']);
        end
    end
end

%--------------------------------------------------------------------------
%subfunction match file channels to directory
function [sortedfiles,sorteddir,ch_num,slice]= ch2dir(chan_dir,curr_files)
% first figure out how many channels there are
ch_num = size(chan_dir,2);  %this is how many channels we expect from the original directory structure
%now parse the files for slice/section number and channel number
%files are of the format "img_000_t1_z1_c5", no type; we will take the last
%file.  No need to be fancy, the file names are stable
last_file = curr_files{end};
seperator = strfind(last_file,'_');
slice = str2double(last_file(seperator(1)+1:seperator(2)-1))+1;
file_ch = str2double(last_file(seperator(end)+2:end));
if ch_num~=file_ch  %sanity check
    warning('File channel number and directory channel number do not match')
end
%now sort the files
sortedfiles = cell(slice,ch_num);  %preallocate
for i = 1:size(curr_files,2)  %go through each file
    curr_file = curr_files{i};
    seperator = strfind(curr_file,'_');
    curr_slice = str2double(curr_file(seperator(1)+1:seperator(2)-1))+1;
    curr_ch = str2double(curr_file(seperator(end)+2:end));
    sortedfiles{curr_slice,curr_ch} = curr_file;
end
%now sort the directory to match channels
ch_idx = cell(1,ch_num);  %preallocate
dir_tmp = horzcat(ch_idx',chan_dir');
for j = 1:ch_num
    curr_ch = chan_dir{j};
    dir_tmp{j,1} = ch_parse(curr_ch);  %each channel is given a place
end
%now sort based on place
dir_tmp = sortrows(dir_tmp,1);
%now output the directory structure
sorteddir = dir_tmp(:,2)';
%--------------------------------------------------------------------------
%subfunction to parse the channels can convert channel to aratome format
function [ch_place] = ch_parse(ch_str)
%couple of things: we are expecting the channel string to be in the formate
%of 'XXXX 350' we are going to use the last wavelength number as a switch
%parse string
wspace = strfind(ch_str,' ');
wavelength = ch_str(wspace(end)+1:end);
%pull characters as well, not using this, but here as an possible future
%feature
%characters = isstrprop(ch_str,'alpha');
%label = ch_str(characters);  %this will take only the letters.
switch wavelength
    case '350'  %this is violet
        ch_place = 2;
    case '488'
        ch_place = 3;
    case {'594','590'}
        ch_place = 4;
    case '647'
        ch_place = 5;
    case '111'  %alignment channel
        ch_place = 1;
    otherwise
        warning('channel not recognized');
end
%--------------------------------------------------------------------------
%subfunction to parse the inputs.
function [ardir,ordir] = parse(input)

ardir = [];
ordir = [];


%Parse the input
if ~isempty(input)
    for i = 1:2:size(input,2)
        if ischar(input{1,i});
            switch input{1,i}
                case 'ardir'
                    ardir = input{1,i+1};
                case 'ordir'
                    ordir = input{1,i+1};
                otherwise
                    warning(['Your input ',input{1,i},' is not recognized.']);
            end
        else
            error(['The parameters you entered is incorrect.  Please check help.']);
        end
    end
end